/* psnap.c - psnap */

/*------------------------------------------------------------------------
 *  psnap  --  print a snapshot of the process table
 *------------------------------------------------------------------------
 */

#include <conf.h>
#include <kernel.h>
#include <proc.h>
#include <dos.h>
#include <q.h>

#define CTRLC 3			/* ASCII code for control-C	*/

#define PRINTF kprintf

struct sttab {
	int state;
	char *descr;
} sttab[9] = {
	PRCURR, "curr",
	PRFREE, "free",
	PRREADY,"rdy",
	PRRECV, "rcvn",
	PRSLEEP, "slp",
	PRSUSP, "susp",
	PRWAIT, "wait",
	PRTRECV, "tout",
	-1, "???"
};

static struct psnap {
	int state;
	int prio;
	char name[10];
	long time;
	char *pbase;
	int plen;
	char *pregs;	// up to here from process table
	int ctx[5];		// context (from stack)
} stab[NPROC];

static struct qent qtab[NQENT];
	
static int pmark[NPROC];
static char qname[4];

void psnap1(int);
void pprint(Bool, int);

#define next(i) qtab[(i)].qnext
#define prev(i) qtab[(i)].qprev
#define key(i)  qtab[(i)].qkey

LOCAL
char * STATE(s)
int s;
{
	struct sttab *sp;

	for (sp = sttab ; sp->state >= 0 ; sp++)
		if (sp->state == s)
			break;
	return(sp->descr);
}

int psnap(Bool showmem)
{
	int i,j;
	int s;
	int ps;

	disable(ps);
	/* retrieve a copy of the process table */
	for ( i = 0; i < NPROC; i++ )
		psnap1(i);
	/* retrieve a copy of the q structure */
	for ( i = 0; i < NQENT; i++ )
		qtab[i] = q[i];
	restore(ps);

	/* now print the process table */
	PRINTF(" id stat prio name       time    ");
	PRINTF(showmem ? "stack limit    stack pointer     stack base\n"
	               : "ss:sp  size   stklim   ds   di   si flag   bp\n");
	for ( i = 0; i < NPROC; i++ )
		pprint(showmem, i);

	if (showmem)
		return;

	PRINTF("Press any key to continue . . .");
	if ( kgetc(0) == CTRLC )
		return;
	PRINTF("\n");

	/* now print the queues */

	for ( i = 0; i < NPROC; i++ )	// initialize pmark[i]
		pmark[i] = 0;
	
	/* Let qprint navigate across semaphore, ready and sleep queues.	*
	 * Each time process i is found, pmark[i] gets incremented			*
	 */
	/* print the semaphore queues (in q struct from entry NPROC upwards)*/
	i = NPROC;
	for ( j = 0; j < NSEM; j++ ) {
		sprintf(qname, "S%d", j);
		qprint(i,qname,PRWAIT,0);
		i += 2;
	}
	qprint(i,"RDY",PRREADY,1);
	i += 2;
	qprint(i,"SLP",PRSLEEP,1);
	
	/* Consistency check: processes not found in any queue shouldn't be	*
	 * in a state implying it should be in a queue!
	 */
	for ( i = 0 ; i < NPROC; i++ )
		if ( pmark[i] == 0 ) {
			s = stab[i].state;
			if ( s == PRWAIT || s == PRREADY || s == PRSLEEP )
				PRINTF("?%02d: %s %s\n",i,stab[i].name,STATE(s));
		}
}

void psnap1(int i)	// call with int disabled
{
	struct pentry *pptr;
	struct psnap *sptr;
	int j, *addr;
	
	pptr = &proctab[i];
	sptr = &stab[i];
	sptr->state = pptr->pstate;
	sptr->prio = pptr->pprio;
	sptr->time = pptr->time;
	strcpy(sptr->name,pptr->pname);
	sptr->pbase = pptr->pbase;
	sptr->plen = pptr->plen;
	sptr->pregs = pptr->pregs;

	/* retrieve process' context from its stack, except for current *
	 */
	if ( pptr->pstate != PRCURR ) {
		addr = (int *) pptr->pregs;
		for ( j = 0 ; j < 5 ; j++ )
			sptr->ctx[j] = *addr++;
	}
}

void pprint(Bool showmem, int i)
{
	int j;
	int s;
	struct pentry *pptr;
	struct psnap *sptr;

	sptr = &stab[i];	// pointer to process entry copied
	if ( (s=sptr->state) != PRFREE ) {
		PRINTF("%3d %-4s %04x %s %5ld ",
		       i, STATE(s), sptr->prio,sptr->name, sptr->time);
		if (showmem) {
			PRINTF("      %05lx          %05lx             %05lx\n", 
			       LINADDR(sptr->pbase), LINADDR(sptr->pregs), 
			       LINADDR((sptr->pbase) + (sptr->plen - 1))  );
			return;
		}
		PRINTF("%P %04x %P", sptr->pregs, sptr->plen, sptr->pbase);
		if ( s != PRCURR )
			for (j = 0; j < 5; j++ )
				PRINTF(" %04x", sptr->ctx[j]);
		PRINTF("\n");
	}
}

/* navigate a queue of processes, with head i, displaying each */
static qprint(i,qnam,state,keyprint)
int i;
char *qnam;
int state;		// all procs in the queue should be in this state
int keyprint;	// print/do not print process priority
{
	int	ni,pi;
	int	s;

	/* entry i,i+1 of qtab should be head/tail of queue */
	if ( prev(i) != EMPTY )
		PRINTF("%s: nonempty qprev header\n",qnam);
	if ( next(i+1) != EMPTY )
		PRINTF("%s: nonempty qnext header\n",qnam);

	for ( ni = next(i), pi = i ; 	// each cycle prints ni, not pi (i=head)
	      ni < NPROC ; 				// print until ni is a process
		  pi = ni, ni = next(ni) ) {//   (stop before queue tail)

  		PRINTF("%s: pid=%02d",qnam,ni);	// print queue name, process id
		if ( ni < 0 ) {
			PRINTF(" ?queue index corrupted!\n");
			break;
		}
		PRINTF(" %s",stab[ni].name);		// print process name
		if ( keyprint )
			PRINTF(" key=%d", key(ni));
		if ( pmark[ni]++ != 0 ) {
		// pmark[ni] is incremented each time ni is 
		// found/printed in a process queue
			PRINTF(" ?process was on another queue!\n"); // shouldn't happen!
			break;
		}
		s = stab[ni].state;
		if ( s != state )	// all queue procs should be in state
			PRINTF(" ?illegal state (%d=%s)",s,STATE(s));
		if ( prev(ni) != pi )	// check consistency of queue entry
			PRINTF(" ?invalid qprev index (%d)",prev(ni));
		PRINTF("\n");
	}
	/* consistency checks for last entry in queue */
	if ( prev(ni) != pi )
		PRINTF("%s: ?invalid qprev index (%d)",qnam,prev(ni));
	if ( ni != i+1 )
		PRINTF("%s: ?queue terminates incorrectly (%d)",qnam,ni);
	/* print separator after non-empty queue */
	if ( pi != i )
		PRINTF("----\n");
}
