/* resched.c - resched */

#include <conf.h>
#include <kernel.h>
#include <proc.h>
#include <dos.h>
#include <q.h>
#include <sleep.h>

/*------------------------------------------------------------------------
 *  resched  --  reschedule processor to highest priority ready process
 *
 * Notes:	Upon entry, currpid gives current process id.
 *			Proctab[currpid].pstate gives correct NEXT state for
 *			current process if other than PRCURR.
 *          I.e., it is the caller's responsibility to have its future
 *			state set before calling resched()
 *------------------------------------------------------------------------
 */

int	resched()
{
	struct pentry  *selfptr;	// pointer to entry of process calling resched
	struct pentry  *nptr;		// pointer to new process entry
	void ctxsw(char * *, char * *);
	int sys_pcxget(void);

	selfptr = &proctab[currpid];
	selfptr->time += 			// increment n. ticks used
		tod - selfptr->oldtime;	// by those of the last run
	
	if ( selfptr->pstate == PRCURR ) {
		/* no switch needed if current prio. higher than next	*/
		if (lastkey (rdytail) < selfptr->pprio ) {
			selfptr->oldtime = tod;
			preempt = QUANTUM;
			return;
        }
        /* or if rescheduling is disabled ( pcxflag == 0 )    */
        if (sys_pcxget() == 0) {
            selfptr->oldtime = tod;
            preempt = 1;	// don't give too much time
            return;
        }
		/* otherwise force context switch */
		selfptr->pstate = PRREADY;
		insert(currpid, rdyhead, selfptr->pprio);
    } 
    else if ( sys_pcxget() == 0 ) {
		/* PANIC: current process decided to change state, but		  *
		 * rescheduling is disabled, so can't even go back to current */
		kprintf("pid=%d state=%d name=%s",
		        currpid, selfptr->pstate, selfptr->pname);
		panic("Reschedule impossible with current process in this state");
	}
	/* OK: proceed to actual context switch */

	/* remove highest priority process at end of ready list */
	nptr = &proctab[(currpid=getlast(rdytail))];
	nptr->pstate = PRCURR;					// mark it currently running
	preempt = QUANTUM;						// reset preemption counter
	nptr->oldtime = tod;					// new current proc starts now

	ctxsw(&selfptr->pregs, &nptr->pregs);	// do context switch
	
	/* Quali sono vecchio e nuovo processo qui? Descriviamo un'esecuzione	*
	 * resched() da parte del processo K. Ci sono 2 punti di vista.			*
	 *																    	*
	 * La chiamata ctxsw() sopra lascia sullo stack di K questo punto come	*
	 * return address per il ret di ctxsw(). Ma ctxsw() cambia stack e il	*
	 * suo ret riporta al ret address sullo stack del nuovo processo J; in	*
	 * linea di principio J potrebbe andare ovunque, ma resched() passa a	*
	 * ctxsw() solo indirizzi &nptr->regs dalla tabella dei processi, 		*
	 * generati o da altre resched() o da create().							*
	 * In conclusione, il ret conclusivo di ctxsw() manda J o al suo entry	*
	 * point o dentro resched(), in questo stesso punto, ma con stack e 	*
	 * contesto diversi rispetto a quelli di K.								*
	 *																		*
	 * Dal punto di vista di K, la chiamata a ctxsw() NON torna subito,		*
	 * cioe` con il ret conclusivo di ctxsw(). Prima o poi qualche processo	*
	 * chiamera` resched() che scegliera` da rdytail proprio K, facendolo	*
	 * currpid. Per quest'ultimo resched(), nptr puntera` a K e il ret 		*
	 * conclusivo del suo ctxsw() ripristinera` lo stack di K: a questo 	*
	 * punto si potra` dire	che K e` tornata dal suo ctxsw().				*
	 *																		*
	 * In conclusione, resched puo` essere letta, prescindendo dal multi-	*
	 * tasking, pensando che ctxsw() e` una chiamata anche lunga ma che al	*
	 * ritorno, quando il processo K che l'ha eseguita riparte, trova tutto	*
	 * invariato, meno currpid che sara` di nuovo K.						*
	 */
	/* Quindi, nel seguito, selfptr punta al processo che ha chiamato		*
	 * resched() e, dopo un certo tempo e` stato scelto di nuovo.			*
	 * Inoltre currpid corrisponde a selfptr.								*
	 */
	if (currpid != NULLPROC && 				// meaningless check for NULLPROC
	    STACK_OFLOW(selfptr))				// Stack overflow is caught 
		panic("stack overflow");			// after it has occurred!
											// Maybe too late!
	if (selfptr->phastrap) {
		selfptr->phastrap = FALSE;				// mark trap as serviced
        if ( (char *) (selfptr->ptfn) != NULL)	// before
			(*selfptr->ptfn)(selfptr->ptarg);	// calling it
	}
	return OK;
}
