; intmap.asm -
; low-level interrupt transfer table and dispatcher 
; common interrupt handling code (proc intcom)

	public	_intmap
	extrn	_is_own_stk:far
	extrn	pcxflag:word		; see xeidi.asm

include ..\h\dos.asm


dseg
STKSIZ		equ	1000h		; max size of system stack
intstack	dw	STKSIZ dup (0)  ; system stack for interrupt handling
topstack	label	byte		; address of (empty) stack top
endds

dseg
_intmap	dd	far ptr intmap		; _intmap punta alla base dell'array intmap
endds					; ma e` accessibile da moduli C (e` in dseg)
					; con il nome intmap (_ non conta in C)
pseg
spsave	dw	?		; saved stack pointer at interrupt
	dw	?		; ??? accessed from C as char * spsave ??? (not yet)
endps

pseg
;--------------------------------------------------------------------------
; intmap  --  hardware interrupt dispatch table
;
; Ogni vettore di interrupt punta a call intcom nella relativa intmap entry
; per questo intmap va dichiarata nel segmento di codice pseg
;--------------------------------------------------------------------------
;
TBLSIZ	equ	20h		; max size of intmap table

intmap	label	byte		; accessibile come intmap da codice C,
				; grazie a _intmap (vedi sopra)
	rept	TBLSIZ		; entry ripetute

	db	0		; ivec - int vector no., used in map.c not here
	call	intcom		; NB: this call is near
	dd      0		; oldisr - old isr from bios (seg:off)
	dd      0		; newisr - new isr code address
	dw      0		; mdevno - minor device number
	dw      0		; bios flag - should BIOS (old) ISR be called?
				; if nonzero, call old ISR
	endm
endps

OLDISR	EQU	cs:dword ptr[bx]	; offset of old ISR within intmap
NEWISR	EQU	cs:dword ptr[bx+4]	; offset of new ISR within intmap
MDEVNO	EQU	cs:word ptr[bx+8]	; offset of minor dev no. within intmap
BIOSFLG	EQU	cs:word ptr[bx+10]	; offset of biosflag within intmap

pseg 				; macro called with bp pointing to frame for 
DONEWISR	macro		; intcom, so [bp+2] is return address for 
	mov	bx,[bp+2]	; "call intcom", so bx points within intmap.
	push	ss		; pass far base pointer (in fact used
	push	bp		; only by ISR exc_isr, ISR for exceptions)
	push	MDEVNO		; pass minor dev. no. as arg
	call	NEWISR		; far call to new (Xinu's) C ISR
	add	sp,6		; pop call's arg (C convention)
	endm			; net effect is to
endps				; call isr(minor-dev-no, far-base-pointer)

pseg

;-------------------------------------------------------------------------
; intcom  --  common hardware interrupt dispatcher
;-------------------------------------------------------------------------
; This procedure is hardware interrupt handling code that is common to all
; hardware interrupt service routines.
;
; It must be reentrant, because although it executes with int flag cleared,
; it calls the C ISR, which could call resched(), which could schedule a 
; process with int flag set, which could be interrupted. 
; intcom would then be re-entered. Hence it should not modify any global, 
; and should use the stack to save data. spsave is an exception to this
; (see below for an explanation).
;
intcom	proc
					; stack top now holds ret address
	push	bp			; which points just after call intcom
	mov	bp,sp			; within intmap entry

	push	ax			; save ax for interrupted code, and
	push	bx			; bx before this ASM code change them
	push	cx			; these could be changed by calls
	push	dx			; to C routines new ISR and is_own_stk
	push	es			; so preserve them for interrupted code

; Certe BIOS ISR vanno chiamate prima (es. KBDINT perche' Xinu legge via BIOS)
; altre non vanno chiamate (es. Ctrl-Break interromperebbe Xinu senza passare
; da xdone() (vedi cbrkint.c))

	mov	bx,[bp+2]	; bx points within intmap, after "call intcom"
	mov	ax,BIOSFLG	; get bios flag from intmap
	or	al,al		; zero?
	jz	ourisr		; yes: skip BIOS ISR, go to our ISR
				; no:  call BIOS ISR

; yes: being a ISR, BIOS ISR should preserve registers: but it does not !
; check when and why (possibly DS is the problem)  
	push	si
	push	di
	push	ds
	pushf			; push flags (simulate int vectored to BIOS)
	call	OLDISR		; call BIOS ISR (returns with IRET)
	cli			; in case BIOS has set interrupt flag
				; (it was disabled by interrupt handled)
	pop	ds
	pop	di
	pop	si
ourisr:

; Con quale stack va eseguita la new ISR? E qual  lo stack a questo punto,
; cio: qual  lo stack nel momento in cui si  verificato l'interrupt?
; E` ancora quello originale, che il processo ha avuto al momento della
; creazione?
;
; Se un processo viene interrotto mentre esegue codice Xinu,
; lo stack pointer punta nello stack originario del processo,
; accessibile via proctab[currpid].pbase.

;	sp,ss were saved to dword cs:spsave here, now moved below

	push	ss		; prepara parametro della call is_own_stk(ss:bp)
	push	bp		; che lascia 0 in ax sse lo stack al momento
	call	_is_own_stk	; di interrupt/ingresso in intcom non  piu`
	add	sp,4		; quello originario del processo interrotto

; NB: is_own_stk could alter BX, but macro DONEWISR will reload it

	or	al,al		; test del valore lasciato da call in ax, se 0
	jz	useSysStack	; passa allo stack di sistema

; can do our ISR without changing stack, since it belongs to current process
	DONEWISR
	jmp	short popregs

; Se invece il processo viene interrotto mentre esegue codice BIOS (ISR BIOS o
; funzione chiamata dal codice Xinu per interagire con hardware),
; Ma il BIOS potrebbe avere cambiato stack (SP e anche SS) e DS. come rivelato
; da is_own_stk.
; Si puo` chiamare new ISR in questo caso?
; DS non e` un problema: la funzione C far new ISR lo inizializza nel prologo.
; Ma e` meglio dare uno stack di sistema (piu` grande?) alla new ISR
; Lo stack corrente viene prima salvato nella double word cs:spsave.

; N.B.: non c'e` il rischio che spsave venga sovrascritto da successive 
; rientri annidati in intcom, perche' la prima volta che si salta a 
; useSysStack, il rescheduling viene disabilitato; 
; quindi il processo corrente (quello che esegue intcom e ha int flag a 0) 
; rimane tale fino al completamento di intcom

useSysStack:	; non si puo` chiamare new ISR con lo stack del BIOS

	mov	cs:spsave,sp		; save current stack (far) pointer
	mov	cs:spsave+2,ss		;
	mov	ax,seg intstack		; get address of alternative stack
	mov	ss,ax			; make it the temporary stack segment
	mov	sp,offset topstack	; make new stack empty

; How much multitasking does Xinu allow during interrupt handling?
; Interrupt flag IF is cleared when hardware interrupt occurs, and still
; cleared (see CLI just before label ourisr:) when new ISR is called.
; However new ISR could call resched(), and the newly scheduled process
; could have interrupts enabled, hence set IF again.
; Thus ISRs could nest, but not indefinitely. 
; For each nesting ISR, starting while a previous ISR is open, must interrupt
; a different process (and use its stack, which limits resource occupation
; due to nested interrupts).
; If there are N pending nested ISRs and N processes, all of them will 
; have interrupts disabled, so any further resched(), whichever process is 
; scheduled, will leave interrupts disabled.

; se l'interrupt gestito qui  avvenuto mentre si era in una routine BIOS, 
; occorre disabilitare il rescheduling, per evitare che si finisca per 
; rientrare nella stessa routine di BIOS (che NON e` rientrante).
; NB: in DOS questo non e` un problema (non c'e` multitasking).
; e` questo che permette l'uso di spsave discusso sopra.

	xor	ax,ax			; disable rescheduling before call to
	xchg	ax, cs:pcxflag		; ISR (which might try to reschedule)
	push	ax			; salva vecchio pcxflag
	DONEWISR			; don't care what new ISR does to regs
	pop	cs:pcxflag		; restore pcxflag
	mov	ss,cs:spsave+2		; restore old stack
	mov	sp,cs:spsave

popregs:
	pop	es			; restore registers
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pop	bp

; ora sul top dello stack c'e` l'indirizzo di ritorno a dopo la chiamata
; a intcom; questa era in una intmap entry, quindi l'indirizzo e` inutile;
	add	sp,2		; remove ret address pushed by call intcom

	iret			; sopra nello stack ci sono invece flags,cs e
		    		; ip del processo che era stato interrotto
intcom	endp			; proprio ci che serve!

endps

	end
