/* initialize.c - main, sysinit */

#include <dos.h>
#include <conf.h>
#include <kernel.h>
#include <mem.h>
#include <io.h>
#include <proc.h>
#include <sem.h>
#include <q.h>
#include <shell.h>
#include <butler.h>
#include <pc.h>
#include <kbdio.h>
#include <bufpool.h>

#ifdef Ntty
#include <tty.h>
int winofcur = 0;			// define current input window
struct tty tty[Ntty];		// window buffers and mode control
#endif

#ifdef Nmf
#include <mffile.h>
struct mfblk mftab[Nmf];
#endif

#ifdef MEMMARK
#include <mark.h>
#endif


/* Declarations of major kernel variables */

struct pentry proctab[NPROC];	// process table
int nextproc;					// next process slot to use in create
struct sentry semaph[NSEM];		// semaphore table
int nextsem; 					// next semaphore slot to use in screate
struct qent q[NQENT];			// q table (see queue.c)
int nextqueue;					// next slot in q structure to use
MBLOCK memlist;					// list of free memory blocks
BLKADDR_T lowbaddr;				// beginning of free memory
BLKADDR_T hibaddr;				// end of free memory
u32_t avmem;					// available memory in bytes

/* PC-specific variables */

int nmaps;			// next intmap entry to be used (see map.c/mapinit())

/* active system state */

int numproc;					// number of live user processes
int currpid;					// id of currently running process
char *lowptr;					// allocated memory start
int run;						// TRUE while Xinu is running

/* real-time clock variables and sleeping process queue pointers */

long tod;						// time-of-day (tics since startup)
long startup;
int defclk;						// non-zero, then deferring clock count
int clkdiff;					// deferred clock ticks
int slnempty;					// FALSE if the sleep queue is empty
int *sltop;						// address of key part of top entry in
								// the sleep queue if slnempty==TRUE
int clockq;						// head of queue of sleeping processes
int preempt;					// preemption counter.  Current process
								// is preempted when it reaches zero;
								// set in resched(); counts in ticks

/* miscellaneous system variables */

int butlerpid;					// pid of butler process
int sysmsg;						// message queue
int rdyhead,rdytail;			// head/tail of ready list (q indexes)

void (* usrpanic)() = &nullfun;	// hook for user defined panic


/************************************************************************/
/***                NOTE:                                             ***/
/***                                                                  ***/
/***   This is where the system begins after the C environment has    ***/
/***   been established.  Interrupts are initially DISABLED, and      ***/
/***   must eventually be enabled explicitly.  This routine turns     ***/
/***   itself into the null process after initialization.  Because    ***/
/***   the null process must always remain ready to run, it cannot    ***/
/***   execute code that might cause it to be suspended, wait for a   ***/
/***   semaphore, or put to sleep, or exit.  In particular, it must   ***/
/***   not do I/O unless it uses kprintf for console output.          ***/
/***                                                                  ***/
/************************************************************************/

/*------------------------------------------------------------------------
 *  main  --  initialize system and become the null process (id==0)
 *------------------------------------------------------------------------
 */
main(argc,argv)					// entry point for EXE file
int argc;
char *argv[];
{
	int xmain();				// user's main procedure
	int butler();				// BUTLER process
	int ps;						// saved processor flags
	int pcx;					// saved reschedule flag
	int psnap(Bool);
	int sysinit(void);

	while ( kbdgetc() != NOCH )	// eat remaining kbd chars
		;
	kprintf("Initializing . . .\n");
	disable(ps);
	xdisable(pcx);		// disable rescheduling for now, note pcxflag
						// is initialized to 1, so pcx is now 1
	if ( sysinit() == SYSERR ) {	// initialize all of PC-Xinu
		kprintf("PC-Xinu initialization error\n");
		maprestore();
		restore(ps);
		return SYSERR;
	}
	kprintf("\nXINU Version %s\n\n", VERSION);
	kprintf("User memory %lu bytes, ", avmem);
	kprintf("from byte %lu ", (u32_t) lowbaddr << BLKDIM);
	kprintf("(address %P) ", lowptr);
	kprintf("to %lu\n", ((u32_t) hibaddr << BLKDIM) - 1);
	kprintf("Null proc stack limit %P\n\n", proctab[NULLPROC].pbase);

	/* Following code may be viewed as executed by the null process,
	 * set up by sysinit() below.
	 */
	restore(ps);
	/* interrupts now enabled, ISRs may trigger, but rescheduling still
	 * disabled, so for now any call to resched brings back to this code
	 */
	gettime(&startup);

	/* allow rescheduling i.e. pcxflag=pcx=1: XINU is now running */
	xrestore(pcx);

	// start the butler process
	butlerpid = create(butler,BTLRSTK,BTLRPRIO,BTLRNAME,0);
	resume(butlerpid);

	/* create the user process */
	resume(create(xmain,0x800,INITPRIO,"xmain",2,argc,argv));

	while (run)
		pause();
	psnap(TRUE);
	return(OK);			// by exiting(), main() caters for
}						// returning to DOS


/*------------------------------------------------------------------------
 *  sysinit  --  initialize all PC-Xinu data structures and devices
 *------------------------------------------------------------------------
 */

LOCAL sysinit()
{
	int i,j;
	char *farmalloc(long);		// C memory allocator
	size_t sizmem;				// memory sizing
	struct pentry *nullptr;
	char *namep;				// null process name pointer
	char *ptr, *currsp;
	struct sentry *sptr;
	struct devsw *dvptr;		// pointer to devtab entry
	int xdone();				// terminate xinu
	char *sys_stknpr(void);

    /*  initialize system variables */

	numproc = 0;
	nextproc = NPROC-1;
	nextsem = NSEM-1;
	nextqueue = NPROC;			// q[0..NPROC-1] are processes
	nmaps = 0;
	run = TRUE;					// run until someone sets run = FALSE

	/* allocate memory */

	avmem = MMAX;					// MMAX is a multiple of BLKSIZ
	do {
		MALLOC(lowptr,avmem);		// try and allocate; BLKSIZ always
		avmem -= BLKSIZ;			// decremented because an extra block
	} while (lowptr == NULL);		// will be trimmed off

	/* initialize memory */

	lowbaddr = ADDR2BLK(lowptr)+1;	// trim off extra block, in case
	lowptr = BLK2ADDR(lowbaddr);	// lowptr did not point at block start
	sizmem = avmem >> BLKDIM;
	hibaddr = lowbaddr + sizmem;	// top of free memory

	memlist.mnext = lowbaddr;		// initialize header of free list
	memlist.mlen = sizmem;
	memnext(lowptr) = 0;
	memlen(lowptr) = sizmem;

	/* initialize process table */

	for (i = 0 ; i < NPROC ; i++)
		proctab[i].pstate = PRFREE;

	/* initialize null process entry */

	nullptr = &proctab[NULLPROC];
	for (ptr = (char *) nullptr;
	     ptr < (char *) (nullptr+1); ptr++)		// zero out
		*ptr = '\0';							// NULLPROC's entry

	nullptr->pstate = PRCURR;
	nullptr->pprio = 0;
	nullptr->oldtime = 0;
	nullptr->time = 0;
	namep = "*NULL*";
	for (j = 0; j < PNMLEN; j++)
		nullptr->pname[j] = (*namep ? *namep++ : ' ');
	nullptr->pname[PNMLEN] = '\0';
	nullptr->paddr = main;
	nullptr->pargs = 0;
#ifdef Nsio
	/* STDIN, STDOUT, STDERR for null process */
	for (i = 0 ; i < Nsio ; i++)
		nullptr->pdevs[i] = (i < 3) ? CONSOLE : BADDEV;
#endif

	/* Memory for null process is on current main()/sysinit() stack!	*
	 * We must initialize pbase and ssize proctab's fields (well be 	*
	 * needed right away, at first interrupt to null process, by 		*
	 * isownstk(). pregs is not needed now (first resched() called by 	*
	 * null process sets it).											*
	 *
	 * It is being assumed that if current sp is XYZW:xyzw, then the	*
	 * stack is free up to XYZW:0000; the stack base should be 			*
	 * XYZW:NULLSTK, but this is really only to give ssize a value, so	*
	 * that isownstk() may work. We could get rid of ssize in the code	*
	 * of isownstk and avoid this assignment here.						*
	 */
	currsp = sys_stknpr();						// get current sp
	nullptr->pbase = currsp - FP_OFF(currsp);	// set stack base
	nullptr->ssize = nullptr->plen = NULLSTK;

	currpid = NULLPROC;

	for (i = 0 ; i < NSEM ; i++) {		// initialize semaphores
		(sptr = &semaph[i])->sstate = SFREE;
		sptr->sqtail = 1 + (sptr->sqhead = newqueue());
	}
    rdytail = 1 + (rdyhead=newqueue());	// initialize ready list

	clkinit();							// initialize real-time clock

#ifdef MEMMARK
	_mkinit();							// initialize memory marking
#else
	pinit();							// initialize ports
	poolinit();							// initialize buffer pools
#endif

#ifdef	Ndsk
	dskdbp = mkpool(DBUFSIZ, NDBUFF);	// initialize disk buffers
	dskrbp = mkpool(DREQSIZ, NDREQ);
#endif

/* set up low-level interrupt vectors, for traps and interrupts, by	*
 * calling mapinit(vector no., ISR, ISR_arg); arg passed to ISR is	*
 * int type again for exceptions, minor device number for devices	*
 */
	mapinit(DB0VEC, exc_isr, DB0VEC);		// divide by zero
	mapinit(SSTEPVEC, exc_isr, SSTEPVEC);	// single step
	mapinit(BKPTVEC, exc_isr, BKPTVEC);		// breakpoint
	mapinit(OFLOWVEC, exc_isr, OFLOWVEC);	// overflow
	mapinit(CBRKVEC, cbrkint, CBRKVEC);		// ctrl-break
	if (mapinit(CLKVEC|BIOSFLG, 			// BIOSFLG to call
	            clkint, CLKVEC) == SYSERR) {		// BIOS ISR, see intmap.asm
		kprintf("initiali.c: error initializing clock\n");
		return(SYSERR);
	}

	/* initialize devices, including interrupt vector if applicable
	 */
#ifdef NDEVS
	for ( i = 0 ; i < NDEVS ; i++ ) {
		dvptr = &devtab[i];
		if ( dvptr->dvivec &&
		     mapinit(dvptr->dvivec, dvptr->dviint,
			         dvptr->dvminor) == SYSERR )
			goto uninit;
		if ( dvptr->dvovec &&
		     mapinit(dvptr->dvovec, dvptr->dvoint,
			         dvptr->dvminor) == SYSERR )
			goto uninit;
		if ( init(i,INIT) == OK )		// initialize the device
			continue;
uninit:
		kprintf("initiali.c: error initializing device %d\n", i);
		while (--i > 0)
			init(i,UNINIT);
		return(SYSERR);
	}
#endif

	return(OK);
}
