/* create.c - create, newpid */

#include <conf.h>
#include <kernel.h>
#include <io.h>
#include <proc.h>
#include <dos.h>
#include <mem.h>

#define INITF 0x0200	// initial flag register - set interrupt flag,
						//   clear direction and trap flags

extern int INITRET();	// location to return to upon termination

/*------------------------------------------------------------------------
 *  create  --  create a process to start running a procedure
 *------------------------------------------------------------------------
 */

SYSCALL create(
intfunptr procaddr,		// process' code address
size_t ssize,			// stack size in bytes
short priority,			// process priority > 0
char *namep,			// name in proctab (useful for debugging)
int nargs,				// number of args that follow
int args,				// arguments to procaddr
...						//   (below args is treated as an array args[])
)
{						
	int pid;			// stores new process id
	struct pentry *pptr;// pointer to proc. table entry
	int i;				// loop variable
	int *currdevs;		// current device table
	int dev;			// device number for stdio
	int *a;             // points to list of args
	char *saddr;		// start of stack address
	int *sp;			// imitates stack pointer
	int ps;				// saved processor status
	int newpid(void);

	disable(ps);

	/* allocate memory */
	ssize = roundp(ssize);	// requested stack size rounded to paragraph
	if (ssize < MINSTK)
		ssize = MINSTK;
	if (priority < 1 || (saddr = getmem(ssize)) == NULL
	    || (pid = newpid()) == SYSERR ) {
		if (saddr != NULL)
			freemem(saddr,ssize);
		restore(ps);
		return(SYSERR);
	}

	numproc++;		// one more process

	/* fill new process' slot in process table */
	pptr = &proctab[pid];
	pptr->pstate = PRSUSP;
	for (i = 0 ; i < PNMLEN ; i++)
		pptr->pname[i] = (*namep ? *namep++ : ' ');
	pptr->pname[PNMLEN] = '\0';
	pptr->pprio = priority;
	pptr->ssize = ssize;		// requested/obtained stack size
	pptr->pbase = saddr;		// set base of process' memory
	pptr->plen = ssize;			// memory only has stack in this version
	pptr->oldtime = 0;			// resched hasn't run this process yet
	pptr->time = 0;				// clear CPU time used
	pptr->phasmsg = FALSE;		// no message
	pptr->pnxtkin = BADPID;		// no next-of-kin
	pptr->phastrap = FALSE;		// no trap yet, but
	pptr->ptfn = die;			// initialize trap to die (see kill())
	pptr->ptarg = 0;			// n. of args for trap
	pptr->paddr = procaddr;	
	pptr->pargs = nargs;
	
	/* new process inherits open I/O devices of parent */
	currdevs = proctab[currpid].pdevs;	// parent's (currpid's) device table
	for (i = 0 ; i < Nsio ; i++) {
		dev = open(currdevs[i], NULL);
		pptr->pdevs[i] = isbaddev(dev) ? BADDEV : dev;
	}

	/* build a stack frame for new process; frame occupies 
	   process' allocated memory thus:
	 */
	sp = (int *)(saddr+ssize);		// simulate stack pointer: point to base
	sp -= SAFETY;					// forse evita piccoli stack overflow?

	a = (&args) + nargs;			// point *past* last argument
	for ( ; nargs > 0 ; nargs--)	// copy args onto
		*(--sp) = *(--a);           //    created process' stack
	*(--((intfunptr *) sp)) =		// push ret address for the return that
		INITRET;					//    will end created process
	*(--((intfunptr *) sp)) = 		// simulate a 
		procaddr;					//    context switch
	--sp;							// 1 word for bp
	*(--sp) = INITF;				// FLAGS value
	sp -= NSAVEREGS;				// words for si, di, ds
	pptr->pregs = (char *)sp;		// save for context switch

	restore(ps);
	return(pid);
}


/*------------------------------------------------------------------------
 *  newpid  --  obtain a new (free) process id (sort of next fit)
 *------------------------------------------------------------------------
 */
LOCAL newpid()
{
    int pid;	// process id to return
	int try;	// n. of slots tested (<max NPROC)

	for (try = 1 ; try <= NPROC ; try++) {	// try NPROC slots max
		if ( (pid = nextproc--) <= 0)		// search backwards, but
			nextproc = NPROC-1;				// modulo NPROC
		if (proctab[pid].pstate == PRFREE)	// found process id/slot
			return(pid);
	}
	return(SYSERR);		// all slots occupied
}