/* resched.c - resched */

#include <conf.h>
#include <kernel.h>
#include <proc.h>
#include <dos.h>
#include <q.h>
#include <sleep.h>

/*------------------------------------------------------------------------
 *  resched  --  reschedule processor to highest priority ready process
 *
 * Notes:	Upon entry, currpid gives current process id.
 *			Proctab[currpid].pstate gives correct NEXT state for
 *			current process if other than PRCURR.
 *          I.e., it is the caller's responsibility to have its future
 *			state set before calling resched()
 *------------------------------------------------------------------------
 */

int	resched()
{
	struct pentry  *optr;  /* pointer to old process entry */
	struct pentry  *nptr;  /* pointer to new process entry */

	optr = &proctab[currpid];
	optr->time += 				// increment n. ticks used
		tod - optr->oldtime;	// by those of the last run
	
	if ( optr->pstate == PRCURR ) {
		/* no switch needed if current prio. higher than next	*/
		if (lastkey ( rdytail) < optr->pprio ) {
			optr->oldtime = tod;
			preempt = QUANTUM;
			return;
        }
        /* or if rescheduling is disabled ( pcxflag == 0 )    */
        if (sys_pcxget() == 0) {
            optr->oldtime = tod;
            preempt = 1;	// don't give too much time
            return;
        }
		/* otherwise force context switch */
		optr->pstate = PRREADY;
		insert(currpid, rdyhead, optr->pprio);
    } 
    else if ( sys_pcxget() == 0 ) {
    		/* PANIC: current process decided to change state, but		  *
    		 * rescheduling is disabled, so can't even go back to current */
			kprintf("pid=%d state=%d name=%s",
			        currpid, optr->pstate, optr->pname);
			panic("Reschedule impossible in this state");
		}

	/* remove highest priority process at end of ready list */
	nptr = &proctab[(currpid=getlast(rdytail))];
	nptr->pstate = PRCURR;		// mark it currently running
	preempt = QUANTUM;			// reset preemption counter
//	_pglob = nptr->pglob;		// retrieve global environment

    nptr->oldtime = tod;				// new current starting now
    ctxsw(&optr->pregs, &nptr->pregs);	// context switch
	/* The OLD process returns here when resumed. */

	if (currpid != 0 && FP_OFF(optr->pregs) > optr->plen)
		panic("stack overflow");
	if (optr->phastrap) {
		optr->phastrap = FALSE;	/* mark trap as serviced	*/
        if (optr->ptfn != NULL)
			(*optr->ptfn)(optr->ptarg);
	}

	return;
}
