; intmap.asm - 
; low-level interrupt transfer table and dispatcher

TBLSIZ	equ	20h			; define max size of intmap table
STKSIZ  equ     1000h                   ; max size of system stack

	public	_sys_imp
	extrn	_get_cur_ss:far

_FARDATA	SEGMENT WORD PUBLIC 'FAR_DATA'
_sys_imp	dd	far ptr intmap
_FARDATA	ENDS

_TEXT	SEGMENT	WORD PUBLIC 'CODE'
	ASSUME CS:_TEXT, DS:NOTHING

	extrn 	pcxflag:word, sssave:word, spsave:word

intstack	db	STKSIZ dup (0)  ; interrupt stack
topstack	label	byte

;-------------------------------------------------------------------------
; intmap  --  hardware interrupt dispatch table
;-------------------------------------------------------------------------
intmap	label	byte			; private, accessed via sys_imp

	rept	TBLSIZ

        db      0                       ; ivec   - interrupt vector number
	call	intcom
        dd      0                       ; oldisr - old isr from bios (seg:off)
        dd      0                       ; newisr - new isr code address
        dw      0                       ; mdevno - minor device number
        dw      0                       ; iflag  - interrupt flag

	endm

;-------------------------------------------------------------------------
; intcom  --  common hardware interrupt dispatcher
;-------------------------------------------------------------------------
; This procedure is hardware interrupt handling code that is common to all
; hardware interrupt service routines.
intcom	proc	near

        push	bp
	mov	bp,sp
	push	ax			; save ax and 
	push	bx			;   bx before they change
        push	si			; push si,di: maybe BIOS ISR 
        push	di			;   changes them

; In fact BIOS ISR should also preserve registers, after all
; but SI and DI have been reported to cause problems.

        mov     bx,[bp+2]		; bx points intmap (past call intcom)
        mov     ax,cs:[bx+10]		; get iflag from intmap
	or	al,al			; zero?
	je	short nobios		; yes: skip call to BIOS ISR
	pushf				; no: push flags to simulate interrupt
	call    cs:dword ptr[bx]	; call BIOS ISR
	cli				; in case BIOS sets interrupts

nobios:
	push	cx			; save rest of registers
	push	dx
	push	ds
	push	es
;	push	0ffffh			; debugging (remove)
	mov	cs:sssave,ss		; save stack (far) pointer
	mov	cs:spsave,sp

; Se un processo viene interrotto mentre esegue codice Xinu, 
; lo stack pointer in sssave/spsave punta nello stack del processo, 
; accessibile via proctab[currpid].pbase.
;
; Se invece il processo viene interrotto mentre esegue BIOS (ISR o no), 
; il BIOS potrebbe avere cambiato stack (anche SS) e DS.
; Si puo` chiamare new ISR in questo caso?
; DS non e` un problema: ogni funzione C far lo inizializza nel prologo.
; Ma e` meglio preparare uno stack locale (piu` grande?)

        push	bx			; preserve bx (points to old ISR)
        call	_get_cur_ss		; da` 0 se stack cambiato
        pop	bx			; restore bx
        or	al,al
        jz	newstack		; set up stack for new ISR

; we can do our ISR, since the stack belongs to the current process

        push	cs:word ptr[bx+8]	; pass minor dev. no.
        call	cs:dword ptr[bx+4]	; call C new ISR (saves si, di)
	add	sp,2			; deallocate parameter (C convention)
        jmp	short popregs

newstack:		; non si puo` chiamare new ISR con lo stack del BIOS
        mov	ax,_TEXT		; get text segment address
        mov	ss,ax			; and set up temporary stack
        mov	sp,offset cs:topstack
        xor	ax,ax			; disable rescheduling
        xchg	ax, cs:pcxflag
        push	ax          		; salva vecchio pcxflag
        push	cs:word ptr[bx+8]	; pass minor dev. no.
        call	cs:dword ptr[bx+4]	; call new ISR (saves si, di)
        add	sp,2			; deallocate parm. (C convention)
        pop	cs:pcxflag
	mov	ss,cs:sssave		; restore old stack
	mov	sp,cs:spsave

popregs:
;	pop	es			; annulla il push 0ffffh (elimina)
	pop	es			; restore all registers
	pop	ds
	pop	dx
	pop	cx
	pop	di
	pop	si
        pop	bx
	pop	ax
	pop	bp
	add	sp,2			; remove pointer to intmap area
	iret
intcom	endp

_TEXT ENDS

	end

